// Mapa interactivo tipo videojuego 2D
// - Movimiento con WASD / flechas
// - Modos de cámara: 1ª, 2ª, 3ª persona
// - Modo vuelo (space)
// - Marcadores persistentes
// - Estado guardado en IndexedDB

(function() {
  'use strict';

  const canvas = document.getElementById('mapCanvas');
  const ctx = canvas.getContext('2d');

  const hudCoords = document.getElementById('hudCoords');
  const hudMode = document.getElementById('hudMode');
  const hudZoom = document.getElementById('hudZoom');
  const hudMarkers = document.getElementById('hudMarkers');
  const dbStatusEl = document.getElementById('dbStatus');

  const modeButtons = Array.from(document.querySelectorAll('.mode-btn'));
  const flyToggle = document.getElementById('flyToggle');
  const addMarkerBtn = document.getElementById('addMarkerBtn');
  const markerLabelInput = document.getElementById('markerLabel');
  const markersList = document.getElementById('markersList');
  const saveStateBtn = document.getElementById('saveStateBtn');
  const resetStateBtn = document.getElementById('resetStateBtn');

  const CAMERA_MODES = {
    FIRST: 'first',
    SECOND: 'second',
    THIRD: 'third'
  };

  const cameraConfig = {
    [CAMERA_MODES.FIRST]: { zoom: 3.0, label: '1ª persona' },
    [CAMERA_MODES.SECOND]: { zoom: 1.8, label: '2ª persona' },
    [CAMERA_MODES.THIRD]: { zoom: 1.0, label: '3ª persona / mapa' }
  };

  let mapImage = new Image();
  mapImage.src = 'assets/map.png';

  const state = {
    player: { x: 400, y: 400 },
    cameraMode: CAMERA_MODES.FIRST,
    flyMode: false,
    zoomOffset: 0,
    markers: [],
  };

  const keys = {};
  let lastTimestamp = performance.now();
  let baseScale = 1;
  let currentTotalScale = 1;

  let saveQueued = false;
  let dbAvailable = true;

  function resizeCanvas() {
    const dpr = window.devicePixelRatio || 1;
    const rect = canvas.getBoundingClientRect();
    canvas.width = rect.width * dpr;
    canvas.height = rect.height * dpr;
    ctx.setTransform(dpr, 0, 0, dpr, 0, 0);
  }

  window.addEventListener('resize', resizeCanvas);
  resizeCanvas();

  // ----- IndexedDB pequeño wrapper -----
  const DB_NAME = 'mapEditorDB';
  const DB_VERSION = 1;
  const STORE_NAME = 'state';
  const STATE_KEY = 'current';

  const dbPromise = (function openDb() {
    return new Promise(function(resolve, reject) {
      if (!('indexedDB' in window)) {
        dbAvailable = false;
        updateDbStatus('IndexedDB no disponible; usando solo memoria.', 'error');
        resolve(null);
        return;
      }

      const request = indexedDB.open(DB_NAME, DB_VERSION);

      request.onupgradeneeded = function(event) {
        const db = event.target.result;
        if (!db.objectStoreNames.contains(STORE_NAME)) {
          db.createObjectStore(STORE_NAME, { keyPath: 'id' });
        }
      };

      request.onsuccess = function(event) {
        const db = event.target.result;
        dbAvailable = true;
        updateDbStatus('Conectado', 'ok');
        resolve(db);
      };

      request.onerror = function() {
        dbAvailable = false;
        updateDbStatus('Error al abrir DB', 'error');
        resolve(null);
      };
    });
  })();

  function updateDbStatus(text, type) {
    if (!dbStatusEl) return;
    dbStatusEl.textContent = 'DB: ' + text;
    dbStatusEl.classList.remove('status-idle', 'status-ok', 'status-error');
    const cls = type === 'ok' ? 'status-ok' : type === 'error' ? 'status-error' : 'status-idle';
    dbStatusEl.classList.add(cls);
  }

  function dbGetState() {
    return dbPromise.then(function(db) {
      if (!db) return null;
      return new Promise(function(resolve) {
        const tx = db.transaction(STORE_NAME, 'readonly');
        const store = tx.objectStore(STORE_NAME);
        const req = store.get(STATE_KEY);
        req.onsuccess = function() {
          resolve(req.result || null);
        };
        req.onerror = function() {
          resolve(null);
        };
      });
    });
  }

  function dbPutState(fullState) {
    return dbPromise.then(function(db) {
      if (!db) return;
      return new Promise(function(resolve) {
        const tx = db.transaction(STORE_NAME, 'readwrite');
        const store = tx.objectStore(STORE_NAME);
        const data = Object.assign({}, fullState, { id: STATE_KEY });
        const req = store.put(data);
        req.onsuccess = function() {
          resolve();
        };
        req.onerror = function() {
          resolve();
        };
      });
    });
  }

  function queueSaveState() {
    if (!dbAvailable) return;
    if (saveQueued) return;
    saveQueued = true;
    updateDbStatus('Pendiente de guardado…', 'idle');
    setTimeout(function() {
      saveQueued = false;
      const exportState = {
        player: state.player,
        cameraMode: state.cameraMode,
        flyMode: state.flyMode,
        zoomOffset: state.zoomOffset,
        markers: state.markers
      };
      dbPutState(exportState).then(function() {
        updateDbStatus('Sincronizado', 'ok');
      });
    }, 350);
  }

  // ----- Marcadores -----
  function addMarker() {
    const label = markerLabelInput.value.trim() || 'Punto sin nombre';
    const id = 'm_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
    state.markers.push({
      id,
      x: state.player.x,
      y: state.player.y,
      label
    });
    markerLabelInput.value = '';
    renderMarkers();
    queueSaveState();
  }

  function deleteMarker(id) {
    const idx = state.markers.findIndex(m => m.id === id);
    if (idx !== -1) {
      state.markers.splice(idx, 1);
      renderMarkers();
      queueSaveState();
    }
  }

  function focusMarker(id) {
    const m = state.markers.find(m => m.id === id);
    if (!m) return;
    state.player.x = m.x;
    state.player.y = m.y;
    queueSaveState();
  }

  function renameMarker(id) {
    const m = state.markers.find(m => m.id === id);
    if (!m) return;
    const newLabel = prompt('Etiqueta del marcador:', m.label);
    if (newLabel && newLabel.trim()) {
      m.label = newLabel.trim();
      renderMarkers();
      queueSaveState();
    }
  }

  function renderMarkers() {
    markersList.innerHTML = '';
    state.markers.forEach(function(m) {
      const li = document.createElement('li');
      li.className = 'marker-item';

      const main = document.createElement('div');
      main.className = 'marker-main';

      const labelSpan = document.createElement('span');
      labelSpan.className = 'marker-label';
      labelSpan.textContent = m.label;

      const coordsSpan = document.createElement('span');
      coordsSpan.className = 'marker-coords';
      coordsSpan.textContent = `(${m.x.toFixed(0)}, ${m.y.toFixed(0)})`;

      main.appendChild(labelSpan);
      main.appendChild(coordsSpan);

      const actions = document.createElement('div');
      actions.className = 'marker-actions';

      const goBtn = document.createElement('button');
      goBtn.textContent = 'Ir';
      goBtn.addEventListener('click', function() {
        focusMarker(m.id);
      });

      const renameBtn = document.createElement('button');
      renameBtn.textContent = 'Renombrar';
      renameBtn.addEventListener('click', function() {
        renameMarker(m.id);
      });

      const delBtn = document.createElement('button');
      delBtn.textContent = 'Borrar';
      delBtn.classList.add('danger');
      delBtn.addEventListener('click', function() {
        deleteMarker(m.id);
      });

      actions.appendChild(goBtn);
      actions.appendChild(renameBtn);
      actions.appendChild(delBtn);

      li.appendChild(main);
      li.appendChild(actions);

      markersList.appendChild(li);
    });

    hudMarkers.textContent = String(state.markers.length);
  }

  // ----- Cámara y HUD -----
  function setCameraMode(mode) {
    if (!CAMERA_MODES[mode.toUpperCase()]) return;
    state.cameraMode = mode;
    modeButtons.forEach(function(btn) {
      const isActive = btn.getAttribute('data-mode') === mode;
      btn.classList.toggle('active', isActive);
    });
    queueSaveState();
  }

  function toggleFlyMode(value) {
    state.flyMode = value;
    flyToggle.checked = state.flyMode;
    queueSaveState();
  }

  function updateHud() {
    hudCoords.textContent = `${state.player.x.toFixed(1)} , ${state.player.y.toFixed(1)}`;
    const conf = cameraConfig[state.cameraMode] || cameraConfig[CAMERA_MODES.FIRST];
    hudMode.textContent = conf.label + (state.flyMode ? ' + Vuelo' : '');
    hudZoom.textContent = currentTotalScale.toFixed(2) + 'x';
    hudMarkers.textContent = String(state.markers.length);
  }

  // ----- Input -----
  window.addEventListener('keydown', function(ev) {
    keys[ev.code] = true;

    if (ev.code === 'Digit1') {
      setCameraMode(CAMERA_MODES.FIRST);
    } else if (ev.code === 'Digit2') {
      setCameraMode(CAMERA_MODES.SECOND);
    } else if (ev.code === 'Digit3') {
      setCameraMode(CAMERA_MODES.THIRD);
    } else if (ev.code === 'Space') {
      ev.preventDefault();
      toggleFlyMode(!state.flyMode);
    } else if (ev.code === 'KeyM') {
      addMarker();
    } else if (ev.code === 'KeyQ') {
      state.zoomOffset = Math.min(state.zoomOffset + 0.2, 2.5);
      queueSaveState();
    } else if (ev.code === 'KeyE') {
      state.zoomOffset = Math.max(state.zoomOffset - 0.2, -0.7);
      queueSaveState();
    }
  });

  window.addEventListener('keyup', function(ev) {
    keys[ev.code] = false;
  });

  modeButtons.forEach(function(btn) {
    btn.addEventListener('click', function() {
      const mode = btn.getAttribute('data-mode');
      setCameraMode(mode);
    });
  });

  flyToggle.addEventListener('change', function() {
    toggleFlyMode(flyToggle.checked);
  });

  addMarkerBtn.addEventListener('click', addMarker);

  saveStateBtn.addEventListener('click', function() {
    const exportState = {
      player: state.player,
      cameraMode: state.cameraMode,
      flyMode: state.flyMode,
      zoomOffset: state.zoomOffset,
      markers: state.markers
    };
    dbPutState(exportState).then(function() {
      updateDbStatus('Guardado manual OK', 'ok');
    });
  });

  resetStateBtn.addEventListener('click', function() {
    if (!confirm('¿Resetear posición, cámara y marcadores?')) return;
    state.player = { x: 400, y: 400 };
    state.cameraMode = CAMERA_MODES.FIRST;
    state.flyMode = false;
    state.zoomOffset = 0;
    state.markers = [];
    renderMarkers();
    setCameraMode(CAMERA_MODES.FIRST);
    toggleFlyMode(false);
    queueSaveState();
  });

  // ----- Main loop -----
  function update(dt) {
    let moveX = 0;
    let moveY = 0;

    if (keys['KeyW'] || keys['ArrowUp']) moveY -= 1;
    if (keys['KeyS'] || keys['ArrowDown']) moveY += 1;
    if (keys['KeyA'] || keys['ArrowLeft']) moveX -= 1;
    if (keys['KeyD'] || keys['ArrowRight']) moveX += 1;

    if (moveX !== 0 || moveY !== 0) {
      const len = Math.hypot(moveX, moveY) || 1;
      moveX /= len;
      moveY /= len;

      const baseSpeed = state.flyMode ? 420 : 260;
      const speed = baseSpeed * dt;

      state.player.x += moveX * speed;
      state.player.y += moveY * speed;

      if (mapImage && mapImage.complete && mapImage.width) {
        const margin = 5;
        state.player.x = Math.max(margin, Math.min(mapImage.width - margin, state.player.x));
        state.player.y = Math.max(margin, Math.min(mapImage.height - margin, state.player.y));
      }

      queueSaveState();
    }
  }

  function render() {
    const displayWidth = canvas.clientWidth;
    const displayHeight = canvas.clientHeight;

    if (!mapImage || !mapImage.complete || !mapImage.width) {
      ctx.fillStyle = '#020617';
      ctx.fillRect(0, 0, displayWidth, displayHeight);
      ctx.fillStyle = '#64748b';
      ctx.font = '12px system-ui';
      ctx.fillText('Cargando mapa…', 12, 20);
      requestAnimationFrame(tick);
      return;
    }

    baseScale = Math.min(
      displayWidth / mapImage.width,
      displayHeight / mapImage.height
    );

    const conf = cameraConfig[state.cameraMode] || cameraConfig[CAMERA_MODES.FIRST];
    const targetScale = baseScale * (conf.zoom + state.zoomOffset);
    currentTotalScale = Math.max(targetScale, baseScale * 0.6);

    const viewWidth = displayWidth / currentTotalScale;
    const viewHeight = displayHeight / currentTotalScale;

    let camX;
    let camY;

    if (state.cameraMode === CAMERA_MODES.THIRD) {
      camX = mapImage.width / 2;
      camY = mapImage.height / 2;
    } else {
      camX = state.player.x;
      camY = state.player.y;
    }

    let sx = camX - viewWidth / 2;
    let sy = camY - viewHeight / 2;

    sx = Math.max(0, Math.min(mapImage.width - viewWidth, sx));
    sy = Math.max(0, Math.min(mapImage.height - viewHeight, sy));

    ctx.clearRect(0, 0, displayWidth, displayHeight);

    ctx.drawImage(
      mapImage,
      sx, sy, viewWidth, viewHeight,
      0, 0, displayWidth, displayHeight
    );

    function worldToScreen(wx, wy) {
      return {
        x: (wx - sx) * currentTotalScale,
        y: (wy - sy) * currentTotalScale,
      };
    }

    // Marcadores
    state.markers.forEach(function(m) {
      const p = worldToScreen(m.x, m.y);
      if (p.x < -5 || p.y < -5 || p.x > displayWidth + 5 || p.y > displayHeight + 5) return;

      ctx.beginPath();
      ctx.arc(p.x, p.y, 6, 0, Math.PI * 2);
      ctx.fillStyle = 'rgba(34,197,94,0.85)';
      ctx.fill();
      ctx.lineWidth = 2;
      ctx.strokeStyle = 'rgba(15,23,42,0.95)';
      ctx.stroke();
    });

    // Jugador
    const playerScreen = worldToScreen(state.player.x, state.player.y);
    ctx.beginPath();
    ctx.arc(playerScreen.x, playerScreen.y, 9, 0, Math.PI * 2);
    ctx.fillStyle = '#0f172a';
    ctx.fill();
    ctx.lineWidth = 2;
    ctx.strokeStyle = '#f9fafb';
    ctx.stroke();

    ctx.beginPath();
    ctx.arc(playerScreen.x, playerScreen.y, 4, 0, Math.PI * 2);
    ctx.fillStyle = '#22c55e';
    ctx.fill();

    updateHud();
  }

  function tick(timestamp) {
    const dt = Math.min((timestamp - lastTimestamp) / 1000, 0.05);
    lastTimestamp = timestamp;
    update(dt);
    render();
    requestAnimationFrame(tick);
  }

  // ----- Cargar estado guardado -----
  function loadInitialState() {
    dbGetState().then(function(dbState) {
      if (dbState) {
        if (dbState.player) {
          state.player = {
            x: dbState.player.x || state.player.x,
            y: dbState.player.y || state.player.y
          };
        }
        if (dbState.cameraMode && CAMERA_MODES[dbState.cameraMode.toUpperCase()]) {
          state.cameraMode = dbState.cameraMode;
        }
        if (typeof dbState.flyMode === 'boolean') {
          state.flyMode = dbState.flyMode;
        }
        if (typeof dbState.zoomOffset === 'number') {
          state.zoomOffset = dbState.zoomOffset;
        }
        if (Array.isArray(dbState.markers)) {
          state.markers = dbState.markers;
        }
      }

      renderMarkers();
      setCameraMode(state.cameraMode);
      toggleFlyMode(state.flyMode);

      requestAnimationFrame(function(ts) {
        lastTimestamp = ts;
        tick(ts);
      });
    });
  }

  if (mapImage.complete) {
    loadInitialState();
  } else {
    mapImage.onload = function() {
      loadInitialState();
    };
  }
})();
