// Mapa interactivo tipo videojuego 2D + editor de arquitectura
// - Movimiento con WASD / flechas
// - Modos de cámara: 1ª, 2ª, 3ª persona
// - Modo vuelo (space)
// - Marcadores persistentes
// - Editor: muros, electricidad, equipos, decoración
// - Estado guardado en IndexedDB

(function() {
  'use strict';

  const canvas = document.getElementById('mapCanvas');
  const ctx = canvas.getContext('2d');

  const hudCoords = document.getElementById('hudCoords');
  const hudMode = document.getElementById('hudMode');
  const hudZoom = document.getElementById('hudZoom');
  const hudMarkers = document.getElementById('hudMarkers');
  const dbStatusEl = document.getElementById('dbStatus');

  const modeButtons = Array.from(document.querySelectorAll('.mode-btn'));
  const flyToggle = document.getElementById('flyToggle');

  const addMarkerBtn = document.getElementById('addMarkerBtn');
  const markerLabelInput = document.getElementById('markerLabel');
  const markersList = document.getElementById('markersList');

  const saveStateBtn = document.getElementById('saveStateBtn');
  const resetStateBtn = document.getElementById('resetStateBtn');

  // Editor DOM
  const editorToolBtns = Array.from(document.querySelectorAll('[data-editor-tool]'));
  const editorToolLabel = document.getElementById('editorToolLabel');
  const editorToolHud = document.getElementById('editorToolHud');
  const clearToolBtn = document.getElementById('clearToolBtn');
  const wallHeightInput = document.getElementById('wallHeight');

  const wallsList = document.getElementById('wallsList');
  const infraList = document.getElementById('infraList');
  const systemsList = document.getElementById('systemsList');
  const decorList = document.getElementById('decorList');

  const CAMERA_MODES = {
    FIRST: 'first',
    SECOND: 'second',
    THIRD: 'third'
  };

  const cameraConfig = {
    [CAMERA_MODES.FIRST]: { zoom: 3.0, label: '1ª persona' },
    [CAMERA_MODES.SECOND]: { zoom: 1.8, label: '2ª persona' },
    [CAMERA_MODES.THIRD]: { zoom: 1.0, label: '3ª persona / mapa' }
  };

  // Definición de herramientas de editor
  const EDIT_TOOLS = {
    wall:    { id: 'wall',    label: 'Muro / división', category: 'wall',    color: '#020617', short: 'W' },
    socket:  { id: 'socket',  label: 'Contacto',        category: 'electric', color: '#fb923c', short: 'C' },
    switch:  { id: 'switch',  label: 'Apagador',        category: 'electric', color: '#facc15', short: 'S' },
    light:   { id: 'light',   label: 'Luz',             category: 'electric', color: '#fde047', short: 'L' },
    fan:     { id: 'fan',     label: 'Ventilador',      category: 'electric', color: '#a5b4fc', short: 'F' },

    monitor: { id: 'monitor', label: 'Monitor',         category: 'systems', color: '#38bdf8', short: 'M' },
    pc:      { id: 'pc',      label: 'PC',              category: 'systems', color: '#0ea5e9', short: 'PC' },
    screen:  { id: 'screen',  label: 'Pantalla',        category: 'systems', color: '#22d3ee', short: 'TV' },
    panel:   { id: 'panel',   label: 'Panel',           category: 'systems', color: '#2dd4bf', short: 'P' },
    mic:     { id: 'mic',     label: 'Micrófono',       category: 'systems', color: '#f97316', short: 'Mic' },
    speaker: { id: 'speaker', label: 'Bocina',          category: 'systems', color: '#a855f7', short: 'Sp' },
    lock:    { id: 'lock',    label: 'Cerradura',       category: 'systems', color: '#facc15', short: 'Lk' },
    motion:  { id: 'motion',  label: 'Sensor mov.',     category: 'systems', color: '#4ade80', short: 'Mo' },
    presence:{ id: 'presence',label: 'Sensor pres.',    category: 'systems', color: '#22c55e', short: 'Pr' },
    camera:  { id: 'camera',  label: 'Cámara',          category: 'systems', color: '#f97373', short: 'Cam' },

    sofa:    { id: 'sofa',    label: 'Sofá',            category: 'decor',   color: '#f9a8d4', short: 'Sf' },
    desk:    { id: 'desk',    label: 'Escritorio',      category: 'decor',   color: '#fbbf24', short: 'Dk' },
    chair:   { id: 'chair',   label: 'Silla',           category: 'decor',   color: '#fed7aa', short: 'Ch' },
    bed:     { id: 'bed',     label: 'Cama',            category: 'decor',   color: '#bef264', short: 'Bd' },
    table:   { id: 'table',   label: 'Mesa',            category: 'decor',   color: '#a3e635', short: 'Tb' },
    shelf:   { id: 'shelf',   label: 'Repisa',          category: 'decor',   color: '#86efac', short: 'Sh' },
  };

  let mapImage = new Image();
  mapImage.src = 'assets/map.png';

  const state = {
    player: { x: 400, y: 400 },
    cameraMode: CAMERA_MODES.FIRST,
    flyMode: false,
    zoomOffset: 0,

    markers: [],
    walls: [],
    items: [],

    editor: {
      activeTool: null,
      wallHeight: 2.6
    }
  };

  // Viewport info para convertir coordenadas
  const view = {
    sx: 0,
    sy: 0,
    scale: 1,
    width: 0,
    height: 0
  };

  const keys = {};
  let lastTimestamp = performance.now();
  let baseScale = 1;
  let currentTotalScale = 1;

  let saveQueued = false;
  let dbAvailable = true;

  // Variables temporales para dibujo de muros
  let isDrawingWall = false;
  let wallStart = null;
  let wallPreviewEnd = null;

  function resizeCanvas() {
    const dpr = window.devicePixelRatio || 1;
    const rect = canvas.getBoundingClientRect();
    canvas.width = rect.width * dpr;
    canvas.height = rect.height * dpr;
    ctx.setTransform(dpr, 0, 0, dpr, 0, 0);
  }

  window.addEventListener('resize', resizeCanvas);
  resizeCanvas();

  // ----- IndexedDB pequeño wrapper -----
  const DB_NAME = 'mapEditorDB_v2';
  const DB_VERSION = 1;
  const STORE_NAME = 'state';
  const STATE_KEY = 'current';

  const dbPromise = (function openDb() {
    return new Promise(function(resolve) {
      if (!('indexedDB' in window)) {
        dbAvailable = false;
        updateDbStatus('IndexedDB no disponible; usando solo memoria.', 'error');
        resolve(null);
        return;
      }

      const request = indexedDB.open(DB_NAME, DB_VERSION);

      request.onupgradeneeded = function(event) {
        const db = event.target.result;
        if (!db.objectStoreNames.contains(STORE_NAME)) {
          db.createObjectStore(STORE_NAME, { keyPath: 'id' });
        }
      };

      request.onsuccess = function(event) {
        const db = event.target.result;
        dbAvailable = true;
        updateDbStatus('Conectado', 'ok');
        resolve(db);
      };

      request.onerror = function() {
        dbAvailable = false;
        updateDbStatus('Error al abrir DB', 'error');
        resolve(null);
      };
    });
  })();

  function updateDbStatus(text, type) {
    if (!dbStatusEl) return;
    dbStatusEl.textContent = 'DB: ' + text;
    dbStatusEl.classList.remove('status-idle', 'status-ok', 'status-error');
    const cls = type === 'ok' ? 'status-ok' : type === 'error' ? 'status-error' : 'status-idle';
    dbStatusEl.classList.add(cls);
  }

  function dbGetState() {
    return dbPromise.then(function(db) {
      if (!db) return null;
      return new Promise(function(resolve) {
        const tx = db.transaction(STORE_NAME, 'readonly');
        const store = tx.objectStore(STORE_NAME);
        const req = store.get(STATE_KEY);
        req.onsuccess = function() {
          resolve(req.result || null);
        };
        req.onerror = function() {
          resolve(null);
        };
      });
    });
  }

  function dbPutState(fullState) {
    return dbPromise.then(function(db) {
      if (!db) return;
      return new Promise(function(resolve) {
        const tx = db.transaction(STORE_NAME, 'readwrite');
        const store = tx.objectStore(STORE_NAME);
        const data = Object.assign({}, fullState, { id: STATE_KEY });
        const req = store.put(data);
        req.onsuccess = function() {
          resolve();
        };
        req.onerror = function() {
          resolve();
        };
      });
    });
  }

  function queueSaveState() {
    if (!dbAvailable) return;
    if (saveQueued) return;
    saveQueued = true;
    updateDbStatus('Pendiente de guardado…', 'idle');
    setTimeout(function() {
      saveQueued = false;
      const exportState = {
        player: state.player,
        cameraMode: state.cameraMode,
        flyMode: state.flyMode,
        zoomOffset: state.zoomOffset,
        markers: state.markers,
        walls: state.walls,
        items: state.items,
        editor: state.editor
      };
      dbPutState(exportState).then(function() {
        updateDbStatus('Sincronizado', 'ok');
      });
    }, 350);
  }

  // ----- Conversión de coordenadas -----
  function worldToScreen(wx, wy) {
    return {
      x: (wx - view.sx) * view.scale,
      y: (wy - view.sy) * view.scale
    };
  }

  function screenToWorld(sx, sy) {
    return {
      x: sx / view.scale + view.sx,
      y: sy / view.scale + view.sy
    };
  }

  // ----- Marcadores -----
  function addMarker() {
    const label = markerLabelInput.value.trim() || 'Punto sin nombre';
    const id = 'm_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
    state.markers.push({
      id,
      x: state.player.x,
      y: state.player.y,
      label
    });
    markerLabelInput.value = '';
    renderMarkers();
    queueSaveState();
  }

  function deleteMarker(id) {
    const idx = state.markers.findIndex(m => m.id === id);
    if (idx !== -1) {
      state.markers.splice(idx, 1);
      renderMarkers();
      queueSaveState();
    }
  }

  function focusMarker(id) {
    const m = state.markers.find(m => m.id === id);
    if (!m) return;
    state.player.x = m.x;
    state.player.y = m.y;
    queueSaveState();
  }

  function renameMarker(id) {
    const m = state.markers.find(m => m.id === id);
    if (!m) return;
    const newLabel = prompt('Etiqueta del marcador:', m.label);
    if (newLabel && newLabel.trim()) {
      m.label = newLabel.trim();
      renderMarkers();
      queueSaveState();
    }
  }

  function renderMarkers() {
    markersList.innerHTML = '';
    state.markers.forEach(function(m) {
      const li = document.createElement('li');
      li.className = 'marker-item';

      const main = document.createElement('div');
      main.className = 'marker-main';

      const labelSpan = document.createElement('span');
      labelSpan.className = 'marker-label';
      labelSpan.textContent = m.label;

      const coordsSpan = document.createElement('span');
      coordsSpan.className = 'marker-coords';
      coordsSpan.textContent = `(${m.x.toFixed(0)}, ${m.y.toFixed(0)})`;

      main.appendChild(labelSpan);
      main.appendChild(coordsSpan);

      const actions = document.createElement('div');
      actions.className = 'marker-actions';

      const goBtn = document.createElement('button');
      goBtn.textContent = 'Ir';
      goBtn.addEventListener('click', function() {
        focusMarker(m.id);
      });

      const renameBtn = document.createElement('button');
      renameBtn.textContent = 'Renombrar';
      renameBtn.addEventListener('click', function() {
        renameMarker(m.id);
      });

      const delBtn = document.createElement('button');
      delBtn.textContent = 'Borrar';
      delBtn.classList.add('danger');
      delBtn.addEventListener('click', function() {
        deleteMarker(m.id);
      });

      actions.appendChild(goBtn);
      actions.appendChild(renameBtn);
      actions.appendChild(delBtn);

      li.appendChild(main);
      li.appendChild(actions);

      markersList.appendChild(li);
    });

    hudMarkers.textContent = String(state.markers.length);
  }

  // ----- Editor: muros e items -----
  function setEditorTool(toolId) {
    if (toolId && !EDIT_TOOLS[toolId]) return;

    state.editor.activeTool = toolId || null;
    editorToolBtns.forEach(function(btn) {
      const id = btn.getAttribute('data-editor-tool');
      btn.classList.toggle('active', id === toolId);
    });

    const label = toolId ? EDIT_TOOLS[toolId].label : 'Ninguna';
    editorToolLabel.textContent = label;
    editorToolHud.textContent = label;

    // reset preview
    isDrawingWall = false;
    wallStart = null;
    wallPreviewEnd = null;

    queueSaveState();
  }

  function clearEditorTool() {
    setEditorTool(null);
  }

  function addWall(startWorld, endWorld) {
    const id = 'w_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
    const height = parseFloat(state.editor.wallHeight) || 2.6;
    state.walls.push({
      id,
      x1: startWorld.x,
      y1: startWorld.y,
      x2: endWorld.x,
      y2: endWorld.y,
      height
    });
    renderWallsList();
    queueSaveState();
  }

  function deleteWall(id) {
    const idx = state.walls.findIndex(w => w.id === id);
    if (idx !== -1) {
      state.walls.splice(idx, 1);
      renderWallsList();
      queueSaveState();
    }
  }

  function focusWall(id) {
    const w = state.walls.find(w => w.id === id);
    if (!w) return;
    // centro del muro
    state.player.x = (w.x1 + w.x2) / 2;
    state.player.y = (w.y1 + w.y2) / 2;
    queueSaveState();
  }

  function addItem(toolId, wx, wy) {
    const def = EDIT_TOOLS[toolId];
    if (!def) return;
    const id = 'i_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
    state.items.push({
      id,
      toolId,
      category: def.category,
      x: wx,
      y: wy
    });
    renderItemsLists();
    queueSaveState();
  }

  function deleteItem(id) {
    const idx = state.items.findIndex(i => i.id === id);
    if (idx !== -1) {
      state.items.splice(idx, 1);
      renderItemsLists();
      queueSaveState();
    }
  }

  function focusItem(id) {
    const it = state.items.find(i => i.id === id);
    if (!it) return;
    state.player.x = it.x;
    state.player.y = it.y;
    queueSaveState();
  }

  function renderWallsList() {
    wallsList.innerHTML = '';
    state.walls.forEach(function(w, index) {
      const li = document.createElement('li');
      li.className = 'editor-item';

      const main = document.createElement('div');
      main.className = 'editor-item-main';

      const labelSpan = document.createElement('span');
      labelSpan.className = 'editor-item-label';
      labelSpan.textContent = 'Muro ' + (index + 1);

      const length = Math.hypot(w.x2 - w.x1, w.y2 - w.y1);
      const metaSpan = document.createElement('span');
      metaSpan.className = 'editor-item-meta';
      metaSpan.textContent = `L=${length.toFixed(0)}px · h=${w.height.toFixed(2)}m`;

      main.appendChild(labelSpan);
      main.appendChild(metaSpan);

      const actions = document.createElement('div');
      actions.className = 'editor-item-actions';

      const goBtn = document.createElement('button');
      goBtn.textContent = 'Ir';
      goBtn.addEventListener('click', function() {
        focusWall(w.id);
      });

      const delBtn = document.createElement('button');
      delBtn.textContent = 'Borrar';
      delBtn.classList.add('danger');
      delBtn.addEventListener('click', function() {
        deleteWall(w.id);
      });

      actions.appendChild(goBtn);
      actions.appendChild(delBtn);

      li.appendChild(main);
      li.appendChild(actions);

      wallsList.appendChild(li);
    });
  }

  function buildItemsList(container, items) {
    container.innerHTML = '';
    items.forEach(function(it) {
      const def = EDIT_TOOLS[it.toolId] || { label: it.toolId };
      const li = document.createElement('li');
      li.className = 'editor-item';

      const main = document.createElement('div');
      main.className = 'editor-item-main';

      const labelSpan = document.createElement('span');
      labelSpan.className = 'editor-item-label';
      labelSpan.textContent = def.label;

      const metaSpan = document.createElement('span');
      metaSpan.className = 'editor-item-meta';
      metaSpan.textContent = `(${it.x.toFixed(0)}, ${it.y.toFixed(0)})`;

      main.appendChild(labelSpan);
      main.appendChild(metaSpan);

      const actions = document.createElement('div');
      actions.className = 'editor-item-actions';

      const goBtn = document.createElement('button');
      goBtn.textContent = 'Ir';
      goBtn.addEventListener('click', function() {
        focusItem(it.id);
      });

      const delBtn = document.createElement('button');
      delBtn.textContent = 'Borrar';
      delBtn.classList.add('danger');
      delBtn.addEventListener('click', function() {
        deleteItem(it.id);
      });

      actions.appendChild(goBtn);
      actions.appendChild(delBtn);

      li.appendChild(main);
      li.appendChild(actions);

      container.appendChild(li);
    });
  }

  function renderItemsLists() {
    const infra = state.items.filter(i => i.category === 'electric');
    const systems = state.items.filter(i => i.category === 'systems');
    const decor = state.items.filter(i => i.category === 'decor');

    buildItemsList(infraList, infra);
    buildItemsList(systemsList, systems);
    buildItemsList(decorList, decor);
  }

  // ----- Cámara y HUD -----
  function setCameraMode(mode) {
    if (!CAMERA_MODES[mode.toUpperCase()]) return;
    state.cameraMode = mode;
    modeButtons.forEach(function(btn) {
      const isActive = btn.getAttribute('data-mode') === mode;
      btn.classList.toggle('active', isActive);
    });
    queueSaveState();
  }

  function toggleFlyMode(value) {
    state.flyMode = value;
    flyToggle.checked = state.flyMode;
    queueSaveState();
  }

  function updateHud() {
    hudCoords.textContent = `${state.player.x.toFixed(1)} , ${state.player.y.toFixed(1)}`;
    const conf = cameraConfig[state.cameraMode] || cameraConfig[CAMERA_MODES.FIRST];
    hudMode.textContent = conf.label + (state.flyMode ? ' + Vuelo' : '');
    hudZoom.textContent = currentTotalScale.toFixed(2) + 'x';
    hudMarkers.textContent = String(state.markers.length);

    const toolId = state.editor.activeTool;
    const label = toolId ? EDIT_TOOLS[toolId].label : 'Ninguna';
    editorToolHud.textContent = label;
  }

  // ----- Input teclado -----
  window.addEventListener('keydown', function(ev) {
    keys[ev.code] = true;

    if (ev.code === 'Digit1') {
      setCameraMode(CAMERA_MODES.FIRST);
    } else if (ev.code === 'Digit2') {
      setCameraMode(CAMERA_MODES.SECOND);
    } else if (ev.code === 'Digit3') {
      setCameraMode(CAMERA_MODES.THIRD);
    } else if (ev.code === 'Space') {
      ev.preventDefault();
      toggleFlyMode(!state.flyMode);
    } else if (ev.code === 'KeyM') {
      addMarker();
    } else if (ev.code === 'KeyQ') {
      state.zoomOffset = Math.min(state.zoomOffset + 0.2, 2.5);
      queueSaveState();
    } else if (ev.code === 'KeyE') {
      state.zoomOffset = Math.max(state.zoomOffset - 0.2, -0.7);
      queueSaveState();
    } else if (ev.code === 'Escape') {
      clearEditorTool();
    }
  });

  window.addEventListener('keyup', function(ev) {
    keys[ev.code] = false;
  });

  // Input ratón para editor
  canvas.addEventListener('mousedown', function(ev) {
    const rect = canvas.getBoundingClientRect();
    const sx = ev.clientX - rect.left;
    const sy = ev.clientY - rect.top;
    const world = screenToWorld(sx, sy);

    const toolId = state.editor.activeTool;

    if (!toolId) return;

    if (toolId === 'wall') {
      isDrawingWall = true;
      wallStart = world;
      wallPreviewEnd = world;
    } else {
      addItem(toolId, world.x, world.y);
    }
  });

  canvas.addEventListener('mousemove', function(ev) {
    if (!isDrawingWall || !wallStart) return;
    const rect = canvas.getBoundingClientRect();
    const sx = ev.clientX - rect.left;
    const sy = ev.clientY - rect.top;
    wallPreviewEnd = screenToWorld(sx, sy);
  });

  canvas.addEventListener('mouseup', function(ev) {
    if (!isDrawingWall || !wallStart) return;
    const rect = canvas.getBoundingClientRect();
    const sx = ev.clientX - rect.left;
    const sy = ev.clientY - rect.top;
    const endWorld = screenToWorld(sx, sy);
    addWall(wallStart, endWorld);
    isDrawingWall = false;
    wallStart = null;
    wallPreviewEnd = null;
  });

  canvas.addEventListener('mouseleave', function() {
    isDrawingWall = false;
    wallStart = null;
    wallPreviewEnd = null;
  });

  // Botones de herramientas
  editorToolBtns.forEach(function(btn) {
    btn.addEventListener('click', function() {
      const toolId = btn.getAttribute('data-editor-tool');
      if (state.editor.activeTool === toolId) {
        clearEditorTool();
      } else {
        setEditorTool(toolId);
      }
    });
  });

  clearToolBtn.addEventListener('click', clearEditorTool);

  wallHeightInput.addEventListener('change', function() {
    const val = parseFloat(wallHeightInput.value);
    if (!isNaN(val)) {
      state.editor.wallHeight = val;
      queueSaveState();
      renderWallsList();
    }
  });

  // Botones de marcadores / sesión
  addMarkerBtn.addEventListener('click', addMarker);

  saveStateBtn.addEventListener('click', function() {
    const exportState = {
      player: state.player,
      cameraMode: state.cameraMode,
      flyMode: state.flyMode,
      zoomOffset: state.zoomOffset,
      markers: state.markers,
      walls: state.walls,
      items: state.items,
      editor: state.editor
    };
    dbPutState(exportState).then(function() {
      updateDbStatus('Guardado manual OK', 'ok');
    });
  });

  resetStateBtn.addEventListener('click', function() {
    if (!confirm('¿Resetear posición, cámara y capas de editor?')) return;
    state.player = { x: 400, y: 400 };
    state.cameraMode = CAMERA_MODES.FIRST;
    state.flyMode = false;
    state.zoomOffset = 0;
    state.markers = [];
    state.walls = [];
    state.items = [];
    state.editor = {
      activeTool: null,
      wallHeight: parseFloat(wallHeightInput.value) || 2.6
    };

    renderMarkers();
    renderWallsList();
    renderItemsLists();
    setCameraMode(CAMERA_MODES.FIRST);
    toggleFlyMode(false);
    clearEditorTool();
    queueSaveState();
  });

  // ----- Main loop -----
  function update(dt) {
    let moveX = 0;
    let moveY = 0;

    if (keys['KeyW'] || keys['ArrowUp']) moveY -= 1;
    if (keys['KeyS'] || keys['ArrowDown']) moveY += 1;
    if (keys['KeyA'] || keys['ArrowLeft']) moveX -= 1;
    if (keys['KeyD'] || keys['ArrowRight']) moveX += 1;

    if (moveX !== 0 || moveY !== 0) {
      const len = Math.hypot(moveX, moveY) || 1;
      moveX /= len;
      moveY /= len;

      const baseSpeed = state.flyMode ? 420 : 260;
      const speed = baseSpeed * dt;

      state.player.x += moveX * speed;
      state.player.y += moveY * speed;

      if (mapImage && mapImage.complete && mapImage.width) {
        const margin = 5;
        state.player.x = Math.max(margin, Math.min(mapImage.width - margin, state.player.x));
        state.player.y = Math.max(margin, Math.min(mapImage.height - margin, state.player.y));
      }

      queueSaveState();
    }
  }

  function render() {
    const displayWidth = canvas.clientWidth;
    const displayHeight = canvas.clientHeight;

    if (!mapImage || !mapImage.complete || !mapImage.width) {
      ctx.fillStyle = '#020617';
      ctx.fillRect(0, 0, displayWidth, displayHeight);
      ctx.fillStyle = '#64748b';
      ctx.font = '12px system-ui';
      ctx.fillText('Cargando mapa…', 12, 20);
      requestAnimationFrame(tick);
      return;
    }

    baseScale = Math.min(
      displayWidth / mapImage.width,
      displayHeight / mapImage.height
    );

    const conf = cameraConfig[state.cameraMode] || cameraConfig[CAMERA_MODES.FIRST];
    const targetScale = baseScale * (conf.zoom + state.zoomOffset);
    currentTotalScale = Math.max(targetScale, baseScale * 0.6);

    const viewWidth = displayWidth / currentTotalScale;
    const viewHeight = displayHeight / currentTotalScale;

    let camX;
    let camY;

    if (state.cameraMode === CAMERA_MODES.THIRD) {
      camX = mapImage.width / 2;
      camY = mapImage.height / 2;
    } else {
      camX = state.player.x;
      camY = state.player.y;
    }

    let sx = camX - viewWidth / 2;
    let sy = camY - viewHeight / 2;

    sx = Math.max(0, Math.min(mapImage.width - viewWidth, sx));
    sy = Math.max(0, Math.min(mapImage.height - viewHeight, sy));

    // Actualizar viewport global
    view.sx = sx;
    view.sy = sy;
    view.scale = currentTotalScale;
    view.width = viewWidth;
    view.height = viewHeight;

    ctx.clearRect(0, 0, displayWidth, displayHeight);

    ctx.drawImage(
      mapImage,
      sx, sy, viewWidth, viewHeight,
      0, 0, displayWidth, displayHeight
    );

    // --- Dibujar muros ---
    ctx.save();
    ctx.lineCap = 'round';
    state.walls.forEach(function(w) {
      const p1 = worldToScreen(w.x1, w.y1);
      const p2 = worldToScreen(w.x2, w.y2);

      ctx.beginPath();
      ctx.moveTo(p1.x, p1.y);
      ctx.lineTo(p2.x, p2.y);
      ctx.lineWidth = 5;
      ctx.strokeStyle = '#020617';
      ctx.stroke();

      ctx.beginPath();
      ctx.moveTo(p1.x, p1.y);
      ctx.lineTo(p2.x, p2.y);
      ctx.lineWidth = 2;
      ctx.strokeStyle = 'rgba(248,250,252,0.95)';
      ctx.stroke();
    });

    // Muro en preview mientras se arrastra
    if (isDrawingWall && wallStart && wallPreviewEnd) {
      const p1 = worldToScreen(wallStart.x, wallStart.y);
      const p2 = worldToScreen(wallPreviewEnd.x, wallPreviewEnd.y);
      ctx.setLineDash([6, 4]);
      ctx.beginPath();
      ctx.moveTo(p1.x, p1.y);
      ctx.lineTo(p2.x, p2.y);
      ctx.lineWidth = 2;
      ctx.strokeStyle = 'rgba(56,189,248,0.9)';
      ctx.stroke();
      ctx.setLineDash([]);
    }

    ctx.restore();

    // --- Dibujar items (electricidad, equipos, decor) ---
    state.items.forEach(function(it) {
      const def = EDIT_TOOLS[it.toolId] || {};
      const p = worldToScreen(it.x, it.y);
      const size = 10;

      ctx.beginPath();
      ctx.arc(p.x, p.y, size, 0, Math.PI * 2);
      ctx.fillStyle = def.color || 'rgba(148,163,184,0.9)';
      ctx.fill();
      ctx.lineWidth = 2;
      ctx.strokeStyle = '#020617';
      ctx.stroke();

      const short = (def.short || '?').toString();
      ctx.fillStyle = '#020617';
      ctx.font = '9px system-ui';
      ctx.textAlign = 'center';
      ctx.textBaseline = 'middle';
      ctx.fillText(short, p.x, p.y + 0.5);
    });

    // --- Jugador ---
    const playerScreen = worldToScreen(state.player.x, state.player.y);
    ctx.beginPath();
    ctx.arc(playerScreen.x, playerScreen.y, 9, 0, Math.PI * 2);
    ctx.fillStyle = '#0f172a';
    ctx.fill();
    ctx.lineWidth = 2;
    ctx.strokeStyle = '#f9fafb';
    ctx.stroke();

    ctx.beginPath();
    ctx.arc(playerScreen.x, playerScreen.y, 4, 0, Math.PI * 2);
    ctx.fillStyle = '#22c55e';
    ctx.fill();

    updateHud();
  }

  function tick(timestamp) {
    const dt = Math.min((timestamp - lastTimestamp) / 1000, 0.05);
    lastTimestamp = timestamp;
    update(dt);
    render();
    requestAnimationFrame(tick);
  }

  // ----- Cargar estado guardado -----
  function loadInitialState() {
    dbGetState().then(function(dbState) {
      if (dbState) {
        if (dbState.player) {
          state.player = {
            x: dbState.player.x || state.player.x,
            y: dbState.player.y || state.player.y
          };
        }
        if (dbState.cameraMode && CAMERA_MODES[dbState.cameraMode.toUpperCase()]) {
          state.cameraMode = dbState.cameraMode;
        }
        if (typeof dbState.flyMode === 'boolean') {
          state.flyMode = dbState.flyMode;
        }
        if (typeof dbState.zoomOffset === 'number') {
          state.zoomOffset = dbState.zoomOffset;
        }
        if (Array.isArray(dbState.markers)) {
          state.markers = dbState.markers;
        }
        if (Array.isArray(dbState.walls)) {
          state.walls = dbState.walls;
        }
        if (Array.isArray(dbState.items)) {
          state.items = dbState.items;
        }
        if (dbState.editor) {
          state.editor = Object.assign({}, state.editor, dbState.editor);
        }
      }

      if (typeof state.editor.wallHeight === 'number') {
        wallHeightInput.value = state.editor.wallHeight.toFixed(2);
      }

      renderMarkers();
      renderWallsList();
      renderItemsLists();
      setCameraMode(state.cameraMode);
      toggleFlyMode(state.flyMode);
      setEditorTool(state.editor.activeTool);

      requestAnimationFrame(function(ts) {
        lastTimestamp = ts;
        tick(ts);
      });
    });
  }

  if (mapImage.complete) {
    loadInitialState();
  } else {
    mapImage.onload = function() {
      loadInitialState();
    };
  }
})();
