// viewer3d.js
// Visor 3D simple para muros, sin WebGL. Proyección en canvas 2D.

(function(){
  const canvas = document.getElementById('view3d');
  const ctx = canvas.getContext('2d');

  const hudCam = document.getElementById('hudCam');
  const hudWalls = document.getElementById('hudWalls');
  const camHeightInput = document.getElementById('camHeight');
  const wallHeightInput = document.getElementById('wallHeight');
  const glassModeInput = document.getElementById('glassMode');
  const floorGridInput = document.getElementById('floorGrid');
  const wallsJsonInput = document.getElementById('wallsJson');
  const loadWallsBtn = document.getElementById('loadWallsBtn');
  const exportWallsBtn = document.getElementById('exportWallsBtn');
  const statusEl = document.getElementById('status');

  let width = 0, height = 0;
  function resize(){
    const rect = canvas.parentElement.getBoundingClientRect();
    width = rect.width;
    height = rect.height;
    canvas.width = width * window.devicePixelRatio;
    canvas.height = height * window.devicePixelRatio;
    ctx.setTransform(window.devicePixelRatio,0,0,window.devicePixelRatio,0,0);
  }
  window.addEventListener('resize', resize);
  resize();

  // Datos de muros
  let walls2D = [];
  let wallHeight = parseFloat(wallHeightInput.value) || 2.6;

  // Cámara
  const camera = {
    x: 0,
    y: parseFloat(camHeightInput.value) || 1.6,
    z: -12,
    yaw: Math.PI * 0.15,
    pitch: -0.22
  };

  const lightDir = normalize({x: 0.6, y: 1.0, z: -0.4});
  const focal = 550;

  function radians(deg){ return deg * Math.PI / 180; }

  function normalize(v){
    const len = Math.hypot(v.x, v.y, v.z) || 1;
    return {x: v.x/len, y: v.y/len, z: v.z/len};
  }

  function projectPoint(px, py, pz){
    // world -> camera space
    let dx = px - camera.x;
    let dy = py - camera.y;
    let dz = pz - camera.z;

    // yaw (Y axis)
    const cosy = Math.cos(camera.yaw);
    const siny = Math.sin(camera.yaw);
    let cx = dx * cosy - dz * siny;
    let cz = dx * siny + dz * cosy;

    // pitch (X axis)
    const cosp = Math.cos(camera.pitch);
    const sinp = Math.sin(camera.pitch);
    let cy = dy * cosp - cz * sinp;
    cz = dy * sinp + cz * cosp;

    if (cz <= 0.1) cz = 0.1;
    const sx = width/2 + (cx / cz) * focal;
    const sy = height/2 - (cy / cz) * (focal * 0.9);
    return {x:sx, y:sy, depth:cz};
  }

  function lerp(a,b,t){ return a+(b-a)*t; }

  function draw(){
    ctx.clearRect(0,0,width,height);

    // fondo
    const grad = ctx.createRadialGradient(
      width/2, height*0.2, 0,
      width/2, height*0.7, height*0.9
    );
    grad.addColorStop(0,'#020617');
    grad.addColorStop(0.5,'#020617');
    grad.addColorStop(1,'#000000');
    ctx.fillStyle = grad;
    ctx.fillRect(0,0,width,height);

    // piso / grid
    if (floorGridInput.checked){
      ctx.save();
      ctx.strokeStyle = 'rgba(148,163,184,0.35)';
      ctx.lineWidth = 1;
      for (let gx=-20; gx<=20; gx++){
        const p1 = projectPoint(gx,0,-20);
        const p2 = projectPoint(gx,0,20);
        ctx.beginPath();
        ctx.moveTo(p1.x,p1.y);
        ctx.lineTo(p2.x,p2.y);
        ctx.stroke();
      }
      for (let gz=-20; gz<=20; gz++){
        const p1 = projectPoint(-20,0,gz);
        const p2 = projectPoint(20,0,gz);
        ctx.beginPath();
        ctx.moveTo(p1.x,p1.y);
        ctx.lineTo(p2.x,p2.y);
        ctx.stroke();
      }
      ctx.restore();
    }

    // convert walls2D (px) a espacio local (metros simplificados)
    // escala: 1 unidad = 100px
    const scale = 1/100;
    const wallMeshes = [];

    walls2D.forEach(w => {
      const x1 = (w.x1 - 512) * scale;
      const z1 = (w.y1 - 512) * scale;
      const x2 = (w.x2 - 512) * scale;
      const z2 = (w.y2 - 512) * scale;
      const h = w.height || wallHeight;

      const pBottomA = projectPoint(x1, 0, z1);
      const pBottomB = projectPoint(x2, 0, z2);
      const pTopA = projectPoint(x1, h, z1);
      const pTopB = projectPoint(x2, h, z2);

      const depth = (pBottomA.depth + pBottomB.depth + pTopA.depth + pTopB.depth)/4;
      wallMeshes.push({
        points: [pBottomA, pBottomB, pTopB, pTopA],
        world: {x1,z1,x2,z2,h},
        depth
      });
    });

    wallMeshes.sort((a,b) => b.depth - a.depth);

    wallMeshes.forEach(mesh => {
      const w = mesh.world;
      const dx = w.x2 - w.x1;
      const dz = w.z2 - w.z1;
      const length = Math.hypot(dx,dz) || 1;
      const nx = -dz / length;
      const nz = dx / length;
      const ny = 0;
      const normal = normalize({x:nx,y:ny,z:nz});
      const light = Math.max(0.1, normal.x*lightDir.x + normal.y*lightDir.y + normal.z*lightDir.z);
      const base = glassModeInput.checked ? [148, 163, 184] : [59,130,246];
      const r = Math.round(base[0]*light);
      const g = Math.round(base[1]*light);
      const b = Math.round(base[2]*light);
      const alpha = glassModeInput.checked ? 0.28 + light*0.25 : 0.9;

      const pts = mesh.points;

      // sombra en piso
      ctx.save();
      ctx.beginPath();
      ctx.moveTo(pts[0].x, pts[0].y);
      ctx.lineTo(pts[1].x, pts[1].y);
      ctx.lineTo(pts[1].x, pts[1].y+18);
      ctx.lineTo(pts[0].x, pts[0].y+18);
      ctx.closePath();
      ctx.fillStyle = 'rgba(15,23,42,0.45)';
      ctx.fill();
      ctx.restore();

      // pared
      ctx.save();
      ctx.beginPath();
      ctx.moveTo(pts[0].x, pts[0].y);
      ctx.lineTo(pts[1].x, pts[1].y);
      ctx.lineTo(pts[2].x, pts[2].y);
      ctx.lineTo(pts[3].x, pts[3].y);
      ctx.closePath();
      ctx.fillStyle = `rgba(${r},${g},${b},${alpha})`;
      ctx.strokeStyle = glassModeInput.checked
        ? 'rgba(226,232,240,0.85)'
        : 'rgba(15,23,42,0.95)';
      ctx.lineWidth = 1.8;
      ctx.fill();
      ctx.stroke();
      ctx.restore();
    });

    hudCam.textContent = `cam: x=${camera.x.toFixed(1)} z=${camera.z.toFixed(1)} yaw=${(camera.yaw*180/Math.PI).toFixed(0)}°`;
    hudWalls.textContent = `muros: ${walls2D.length}`;
  }

  // input / órbita
  let dragging = false;
  let lastX = 0, lastY = 0;

  canvas.addEventListener('mousedown', e=>{
    dragging = true;
    lastX = e.clientX;
    lastY = e.clientY;
  });
  window.addEventListener('mouseup', ()=> dragging=false);
  window.addEventListener('mousemove', e=>{
    if(!dragging) return;
    const dx = e.clientX - lastX;
    const dy = e.clientY - lastY;
    lastX = e.clientX;
    lastY = e.clientY;
    camera.yaw += dx * -0.005;
    camera.pitch += dy * -0.004;
    camera.pitch = Math.max(-1.2, Math.min(0.3, camera.pitch));
  });

  canvas.addEventListener('wheel', e=>{
    e.preventDefault();
    const delta = e.deltaY>0 ? 1 : -1;
    camera.z += delta * 1.2;
    camera.z = Math.max(-30, Math.min(-4, camera.z));
  }, {passive:false});

  window.addEventListener('keydown', e=>{
    const speed = 0.35;
    if(e.key === 'w' || e.key === 'W'){
      camera.x += Math.sin(camera.yaw) * speed;
      camera.z += Math.cos(camera.yaw) * speed;
    }
    if(e.key === 's' || e.key === 'S'){
      camera.x -= Math.sin(camera.yaw) * speed;
      camera.z -= Math.cos(camera.yaw) * speed;
    }
    if(e.key === 'a' || e.key === 'A'){
      camera.x -= Math.cos(camera.yaw) * speed;
      camera.z += Math.sin(camera.yaw) * speed;
    }
    if(e.key === 'd' || e.key === 'D'){
      camera.x += Math.cos(camera.yaw) * speed;
      camera.z -= Math.sin(camera.yaw) * speed;
    }
  });

  camHeightInput.addEventListener('input', ()=>{
    camera.y = parseFloat(camHeightInput.value) || 1.6;
  });
  wallHeightInput.addEventListener('input', ()=>{
    wallHeight = parseFloat(wallHeightInput.value) || 2.6;
    walls2D.forEach(w=>{ w.height = wallHeight; });
  });

  glassModeInput.addEventListener('change', ()=>{});
  floorGridInput.addEventListener('change', ()=>{});

  loadWallsBtn.addEventListener('click', ()=>{
    try{
      const raw = wallsJsonInput.value.trim();
      if(!raw){
        statusEl.textContent = 'No hay JSON para cargar.';
        return;
      }
      const data = JSON.parse(raw);
      if(!Array.isArray(data)) throw new Error('Debe ser un arreglo []');
      walls2D = data.map(w=>({
        x1: Number(w.x1)||0,
        y1: Number(w.y1)||0,
        x2: Number(w.x2)||0,
        y2: Number(w.y2)||0,
        height: Number(w.height)||wallHeight
      }));
      statusEl.textContent = 'Muros cargados: '+walls2D.length;
    }catch(err){
      statusEl.textContent = 'Error al parsear JSON: '+err.message;
    }
  });

  exportWallsBtn.addEventListener('click', ()=>{
    const json = JSON.stringify(walls2D, null, 2);
    wallsJsonInput.value = json;
    statusEl.textContent = 'Muros exportados a textarea.';
  });

  // demo simple
  const demoWalls = [
    {x1:300,y1:300,x2:700,y2:300,height:2.6},
    {x1:700,y1:300,x2:700,y2:700,height:2.6},
    {x1:700,y1:700,x2:300,y2:700,height:2.6},
    {x1:300,y1:700,x2:300,y2:300,height:2.6},
    {x1:300,y1:500,x2:500,y2:500,height:2.6}
  ];
  walls2D = demoWalls.slice();
  wallsJsonInput.value = JSON.stringify(demoWalls,null,2);
  statusEl.textContent = 'Demo inicial cargada.';

  function loop(){
    draw();
    requestAnimationFrame(loop);
  }
  loop();
})();
