<?php
declare(strict_types=1);

function json_out(array $data, int $code=200): void {
  http_response_code($code);
  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
  exit;
}

function read_json_body(): array {
  $raw = file_get_contents('php://input');
  if ($raw === false || trim($raw) === '') return [];
  $d = json_decode($raw, true);
  return is_array($d) ? $d : [];
}

function safe_id(string $s): string {
  $s = preg_replace('/[^a-zA-Z0-9_\-]/', '_', $s);
  return substr($s ?? '', 0, 64);
}

function now_ms(): int {
  return (int) round(microtime(true) * 1000);
}

function data_dir(): string {
  $dir = __DIR__ . '/data';
  if (!is_dir($dir)) mkdir($dir, 0777, true);
  return $dir;
}

function uploads_dir(): string {
  $dir = __DIR__ . '/uploads';
  if (!is_dir($dir)) mkdir($dir, 0777, true);
  return $dir;
}

function load_json_file(string $path): array {
  if (!file_exists($path)) return [];
  $d = json_decode((string)file_get_contents($path), true);
  return is_array($d) ? $d : [];
}

function save_json_file(string $path, array $data): void {
  file_put_contents($path, json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
}

function load_cfg(string $name): array {
  $path = dirname(__DIR__) . '/config/' . $name;
  return load_json_file($path);
}

function session_path(string $session_id): string {
  return data_dir() . '/session_' . safe_id($session_id) . '.json';
}

function events_path(string $session_id): string {
  return data_dir() . '/events_' . safe_id($session_id) . '.jsonl';
}

function new_session_id(): string {
  $t = gmdate('Ymd_His');
  $rnd = bin2hex(random_bytes(3));
  return "mk_" . $t . "_" . $rnd;
}

function append_event(string $session_id, string $kind, string $source, array $payload): void {
  $evt = [
    'v' => 1,
    'ts' => now_ms(),
    'session_id' => $session_id,
    'kind' => $kind,
    'source' => $source,
    'payload' => $payload
  ];
  file_put_contents(events_path($session_id), json_encode($evt, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) . "\n", FILE_APPEND);
}

function load_session(string $session_id): array {
  $p = session_path($session_id);
  $S = load_json_file($p);
  return is_array($S) ? $S : [];
}

function save_session(array $S): void {
  if (!isset($S['session_id'])) return;
  $S['last_update'] = now_ms();
  save_json_file(session_path((string)$S['session_id']), $S);
}

function clamp(float $v, float $min, float $max): float {
  return max($min, min($max, $v));
}

function derive_completion(array $answers, array $questions): float {
  $keys = [];
  foreach (($questions['items'] ?? []) as $q) {
    if (isset($q['key'])) $keys[] = (string)$q['key'];
  }
  $total = count($keys);
  if ($total <= 0) return 0.0;
  $filled = 0;
  foreach ($keys as $k) {
    if (isset($answers[$k]) && is_array($answers[$k]) && trim((string)($answers[$k]['value'] ?? '')) !== '') $filled++;
  }
  return round(100.0 * ($filled / $total), 2);
}

function derive_zero_shot(array $S, array $branches_cfg, array $policy): array {
  $answers = $S['answers'] ?? [];
  $income = (float)($answers['cual_es_su_ingreso_promedio_mensual_aproximado']['value'] ?? 0);
  $expenses = (float)($answers['cuales_son_sus_gastos_mensuales_aproximados']['value'] ?? 0);
  $months = (int)($answers['cuanto_tiempo_tiene_operando_su_negocio']['value'] ?? 0);
  $dist_km = (float)($answers['distancia_km_a_sucursal']['value'] ?? 0);

  $ratio = ($income > 0) ? (($income - $expenses) / $income) : 0.0;
  $s_cash = clamp(0.10 + 0.90 * $ratio, 0.0, 1.0);

  $s_stab = 0.30;
  if ($months >= 6 && $months < 12) $s_stab = 0.50;
  else if ($months >= 12 && $months < 24) $s_stab = 0.70;
  else if ($months >= 24) $s_stab = 0.85;

  $branch_id = (string)($S['branch_id'] ?? '');
  $fit = 65.0;
  foreach (($branches_cfg['items'] ?? []) as $b) {
    if ((string)($b['branch_id'] ?? '') === $branch_id) {
      $fit = (float)($b['fit_score'] ?? 65.0);
      break;
    }
  }
  $s_ctx = clamp($fit / 100.0, 0.0, 1.0);

  $radius_km = (float)($policy['radius_rules']['branch_radius_km_default'] ?? 10);
  $s_dist = 1.0;
  if ($dist_km > 0 && $radius_km > 0) {
    if ($dist_km <= $radius_km) $s_dist = 1.0;
    else $s_dist = clamp(1.2 - ($dist_km / $radius_km), 0.2, 1.0);
  }

  $score = 0.40*$s_cash + 0.25*$s_stab + 0.20*$s_ctx + 0.15*$s_dist;
  $score = clamp($score, 0.0, 1.0);

  $seg = 'BAJO_POTENCIAL';
  if ($score >= 0.80) $seg = 'ALTO_POTENCIAL';
  else if ($score >= 0.60) $seg = 'MEDIO_POTENCIAL';

  return [
    'score' => round($score, 4),
    'segment' => $seg,
    'components' => [
      'cashflow' => round($s_cash, 4),
      'stability' => round($s_stab, 4),
      'context' => round($s_ctx, 4),
      'distance' => round($s_dist, 4)
    ],
    'version' => 'zs_2026_01'
  ];
}
